"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function buildQueue(scope, id, props) {
    // If props is undefined, define it
    props = (props === undefined) ? {} : props;
    // Conditional queue creation
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Deploy the queue
        return deployQueue(scope, id, props.queueProps, props.deadLetterQueue);
        // If an existingQueueObj is specified, return that object as the queue to be used
    }
    else {
        return props.existingQueueObj;
    }
}
exports.buildQueue = buildQueue;
function deployQueue(scope, id, queuePropsParam, deadLetterQueueParam) {
    // Setup the queue
    let queueProps;
    if (queuePropsParam) {
        // If property overrides have been provided, incorporate them and deploy
        queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), queuePropsParam);
    }
    else {
        // If no property overrides, deploy using the default configuration
        queueProps = defaults.DefaultQueueProps();
    }
    // Determine whether a DLQ property should be added
    if (deadLetterQueueParam) {
        queueProps.deadLetterQueue = deadLetterQueueParam;
    }
    const queue = new sqs.Queue(scope, id, queueProps);
    applySecureQueuePolicy(queue);
    // Return the queue
    return queue;
}
function buildDeadLetterQueue(props) {
    const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
    // Setup the queue interface
    const dlq = {
        maxReceiveCount: mrc,
        queue: props.deadLetterQueue
    };
    // Return the dead letter queue
    return dlq;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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