"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
function buildS3Bucket(scope, props, bucketId) {
    // Conditional s3 Bucket creation
    if (!props.existingBucketObj) {
        if (props.bucketProps) {
            return s3BucketWithLogging(scope, props.bucketProps, bucketId);
        }
        else {
            return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
        }
    }
    else {
        return [props.existingBucketObj];
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        bucketprops = s3_bucket_defaults_1.DefaultS3Props();
    }
    else {
        // Create the Logging Bucket
        loggingBucket = new s3.Bucket(scope, _loggingBucketId, s3_bucket_defaults_1.DefaultS3Props());
        applySecureBucketPolicy(loggingBucket);
        // Extract the CfnBucket from the loggingBucket
        const loggingBucketResource = loggingBucket.node.findChild('Resource');
        // Override accessControl configuration and add metadata for the logging bucket
        loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
        loggingBucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W35',
                        reason: `This S3 bucket is used as the access logging bucket for another bucket`
                    }]
            }
        };
        bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
//# sourceMappingURL=data:application/json;base64,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