"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId) {
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let _lambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X) {
        _lambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionPropsForNodeJS(lambdaServiceRole), lambdaFunctionProps);
    }
    const lambdafunction = new lambda.Function(scope, _functionId, _lambdaFunctionProps);
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with more tighter permissions.`
                }]
        }
    };
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
//# sourceMappingURL=data:application/json;base64,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