"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Files = void 0;
const s3assets = require("@aws-cdk/aws-s3-assets");
/**
 * Additional files to use in a shell script
 */
class Files {
    /**
     * Use the files from a CodePipeline artifact
     */
    static fromArtifact(artifact) {
        if (!artifact) {
            // Typechecking may mess up
            throw new Error('Files.fromArtifact(): input artifact is required, got undefined');
        }
        return {
            bind: () => ({ artifact }),
            grantRead: () => { },
        };
    }
    /**
     * Create a new asset to bundle up the files in a directory on disk
     */
    static fromDirectory(directoryPath) {
        let realFiles;
        return {
            bind(scope) {
                realFiles = Files.fromAsset(new s3assets.Asset(scope, directoryPath, {
                    path: directoryPath,
                }));
                return realFiles.bind(scope);
            },
            grantRead(grantee) {
                if (!realFiles) {
                    throw new Error('bind() must be called first');
                }
                realFiles.grantRead(grantee);
            },
        };
    }
    /**
     * Use an existing asset as a file source
     */
    static fromAsset(asset) {
        return {
            bind: () => ({
                commands: [
                    `echo "Downloading additional files from ${asset.s3ObjectUrl}"`,
                    `aws s3 cp ${asset.s3ObjectUrl} /tmp/dl.zip`,
                    'unzip /tmp/dl.zip -d .',
                ],
            }),
            grantRead: (grantee) => asset.grantRead(grantee),
        };
    }
    constructor() {
    }
}
exports.Files = Files;
//# sourceMappingURL=data:application/json;base64,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