"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const crypto = require("crypto");
const path = require("path");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const core_1 = require("@aws-cdk/core");
const construct_internals_1 = require("../private/construct-internals");
const fs_1 = require("../private/fs");
const _util_1 = require("./_util");
/**
 * (experimental) A standard synth with a generated buildspec.
 *
 * @experimental
 */
class SimpleSynthAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * (experimental) Create a standard NPM synth action.
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     *
     * @experimental
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Create a standard Yarn synth action.
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     *
     * @experimental
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * (experimental) Project generated to run the synth command.
     *
     * @experimental
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: _util_1.filterEmpty([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                    ]),
                },
                build: {
                    commands: _util_1.filterEmpty([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environment = { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment };
        const environmentVariables = {
            ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
            ...this.props.environmentVariables,
        };
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: buildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_d = this.props.additionalArtifacts) !== null && _d !== void 0 ? _d : []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            var _a;
            const cloudAsmArtifactSpec = {
                'base-directory': fs_1.toPosixPath(path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope))),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': fs_1.toPosixPath(path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory)),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * (experimental) The CodeBuild Project's principal.
     *
     * @experimental
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    var _a, _b, _c, _d, _e;
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: (_a = env.buildImage) === null || _a === void 0 ? void 0 : _a.type,
        imageId: (_b = env.buildImage) === null || _b === void 0 ? void 0 : _b.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: (_c = env.buildImage) === null || _c === void 0 ? void 0 : _c.imagePullPrincipalType,
        secretsManagerArn: (_e = (_d = env.buildImage) === null || _d === void 0 ? void 0 : _d.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.secretArn,
    };
}
//# sourceMappingURL=data:application/json;base64,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