"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const core_1 = require("@aws-cdk/core");
const actions_1 = require("./actions");
const asset_manifest_1 = require("./private/asset-manifest");
const toposort_1 = require("./private/toposort");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) Stage in a CdkPipeline.
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 *
 * @experimental
 */
class CdkStage extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
        core_1.Aspects.of(this).add({ visit: () => this.prepareStage() });
    }
    /**
     * (experimental) Add all stacks in the application Stage to this stage.
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     *
     * @experimental
     */
    addApplication(appStage, options = {}) {
        var _a;
        const asm = appStage.synth();
        const extraRunOrderSpace = (_a = options.extraRunOrderSpace) !== null && _a !== void 0 ? _a : 0;
        if (asm.stacks.length === 0) {
            // If we don't check here, a more puzzling "stage contains no actions"
            // error will be thrown come deployment time.
            throw new Error(`The given Stage construct ('${appStage.node.path}') should contain at least one Stack`);
        }
        const sortedTranches = toposort_1.topologicalSort(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(extraRunOrderSpace + 2); // 2 actions for Prepare/Execute ChangeSet
            let executeRunOrder = runOrder + extraRunOrderSpace + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: runOrder + 1 });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * (experimental) Add a deployment action based on a stack artifact.
     *
     * @experimental
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        var _a, _b;
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(2);
        const executeRunOrder = (_b = options.executeRunOrder) !== null && _b !== void 0 ? _b : runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * (experimental) Add a manual approval action.
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     *
     * @experimental
     */
    addManualApprovalAction(options = {}) {
        var _a;
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(),
        }));
    }
    /**
     * (experimental) Add one or more CodePipeline Actions.
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     *
     * @experimental
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * (experimental) Return the runOrder number necessary to run the next Action in sequence with the rest.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     *
     * @experimental
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * (experimental) Whether this Stage contains an action to deploy the given stack, identified by its artifact ID.
     *
     * @experimental
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepareStage() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder, stackArtifact, executeRunOrder } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder,
                executeRunOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = actions_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publishg the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = actions_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                });
            }
        }
    }
}
exports.CdkStage = CdkStage;
/**
 * (experimental) A single output of a Stack.
 *
 * @experimental
 */
class StackOutput {
    /**
     * (experimental) Build a StackOutput from a known artifact and an output name.
     *
     * @experimental
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,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