"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) A Pipeline to deploy CDK apps.
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @experimental
 */
class CdkPipeline extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        if ((_a = props.selfMutating) !== null && _a !== void 0 ? _a : true) {
            this._pipeline.addStage({
                stageName: 'UpdatePipeline',
                actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                        cloudAssemblyInput: this._cloudAssemblyArtifact,
                        pipelineStackName: pipelineStack.stackName,
                        cdkCliVersion: props.cdkCliVersion,
                        projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                    })],
            });
        }
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
        });
    }
    /**
     * (experimental) The underlying CodePipeline object.
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     *
     * @experimental
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * (experimental) Access one of the pipeline's stages by stage name.
     *
     * You can use this to add more Actions to a stage.
     *
     * @experimental
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * (experimental) Add pipeline stage that will deploy the given application stage.
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     *
     * @experimental
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * (experimental) Add a new, empty stage to the pipeline.
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     *
     * @experimental
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * (experimental) Get the StackOutput object that holds this CfnOutput's value in this pipeline.
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     *
     * @experimental
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * (experimental) Validate that we don't have any stacks violating dependency order in the pipeline.
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     *
     * @experimental
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        // CodePipelines has a hard limit of 50 actions per stage. See https://github.com/aws/aws-cdk/issues/9353
        this.MAX_PUBLISHERS_PER_STAGE = 50;
        this.publishers = {};
        this.assetRoles = {};
        this.stages = [];
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        this.pipeline = this.props.pipeline;
        // Hacks to get access to the innards of Pipeline
        const stages = this.props.pipeline._stages;
        // Any asset publishing stages will be added directly after the last stage that currently exists.
        this.lastStageBeforePublishing = stages.slice(-1)[0];
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        var _a;
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // Dynamically create new stages as needed, with `MAX_PUBLISHERS_PER_STAGE` assets per stage.
            const stageIndex = Math.floor((this._fileAssetCtr + this._dockerAssetCtr) / this.MAX_PUBLISHERS_PER_STAGE);
            if (stageIndex >= this.stages.length) {
                const previousStage = (_a = this.stages.slice(-1)[0]) !== null && _a !== void 0 ? _a : this.lastStageBeforePublishing;
                this.stages.push(this.pipeline.addStage({
                    stageName: `Assets${stageIndex > 0 ? stageIndex + 1 : ''}`,
                    placement: { justAfter: previousStage },
                }));
            }
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${++this._fileAssetCtr}` : `DockerAsset${++this._dockerAssetCtr}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
            });
            this.stages[stageIndex].addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
                'codebuild:BatchPutCodeCoverages',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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