"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) Type of the asset that is being published.
 *
 * @experimental
 */
var AssetType;
(function (AssetType) {
    AssetType["FILE"] = "file";
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * (experimental) Action to publish an asset in the pipeline.
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 *
 * @experimental
 */
class PublishAssetsAction extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                privileged: (props.assetType === AssetType.DOCKER_IMAGE) ? true : undefined,
            },
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.list({ produce: () => this.commands }),
                    },
                },
            }),
            role: props.role,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
            role: props.role,
            // Add this purely so that the pipeline will selfupdate if the CLI version changes
            environmentVariables: props.cdkCliVersion ? {
                CDK_CLI_VERSION: { value: props.cdkCliVersion },
            } : undefined,
        });
    }
    /**
     * (experimental) Add a single publishing command.
     *
     * Manifest path should be relative to the root Cloud Assembly.
     *
     * @experimental
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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