"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const fs = require("fs");
const path = require("path");
const cfn = require("@aws-cdk/aws-cloudformation");
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const construct_internals_1 = require("../private/construct-internals");
const fs_1 = require("../private/fs");
/**
 * (experimental) Action to deploy a CDK Stack.
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 *
 * @experimental
 */
class DeployCdkStackAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
            templateConfiguration: props.templateConfigurationPath ? props.cloudAssemblyInput.atPath(props.templateConfigurationPath) : undefined,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * (experimental) Construct a DeployCdkStackAction from a Stack artifact.
     *
     * @experimental
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = artifact.environment.account;
        const account = artAccount === core_1.Stack.of(scope).account || artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        const actionRole = roleFromPlaceholderArn(scope, region, account, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, region, account, artifact.cloudFormationExecutionRoleArn);
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        let fullConfigPath;
        if (Object.keys(artifact.tags).length > 0) {
            fullConfigPath = `${fullTemplatePath}.config.json`;
            // Write the template configuration file (for parameters into CreateChangeSet call that
            // cannot be configured any other way). They must come from a file, and there's unfortunately
            // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
            // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
            writeTemplateConfiguration(fullConfigPath, {
                Tags: artifact.tags,
            });
        }
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath: fs_1.toPosixPath(path.relative(appAsmRoot, fullTemplatePath)),
            templateConfigurationPath: fullConfigPath ? fs_1.toPosixPath(path.relative(appAsmRoot, fullConfigPath)) : undefined,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, region, account, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = constructs_1.Node.of(scope).tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = constructs_1.Node.of(scope).tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
        region: region !== null && region !== void 0 ? region : core_1.Aws.REGION,
        accountId: account !== null && account !== void 0 ? account : core_1.Aws.ACCOUNT_ID,
        partition: core_1.Aws.PARTITION,
    });
    return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false });
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
/**
 * Write template configuration to the given file
 */
function writeTemplateConfiguration(filename, config) {
    fs.writeFileSync(filename, JSON.stringify(config, undefined, 2), { encoding: 'utf-8' });
}
//# sourceMappingURL=data:application/json;base64,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