""" Functions to create S104 files and populate with data from other sources

"""
import logging
import sys
import datetime

import numpy

from ..s1xx import s1xx_sequence
from .api import S104File, FILLVALUE_HEIGHT, FILLVALUE_TREND, S104Exception


def _get_S104File(output_file):
    """ Small helper function to convert the output_file parameter into a S104File"""
    if isinstance(output_file, S104File):
        data_file = output_file
    else:
        try:
            data_file = S104File(output_file, "w")
        except TypeError as typeerr:
            msg = "Failed to create S104File using {}".format(str(output_file))
            logging.error(msg)
            raise type(typeerr)(msg).with_traceback(sys.exc_info()[2])

    return data_file


def create_s104(output_file) -> S104File:
    """ Creates or updates an S104File object.
    Default values are set for any data that doesn't have options or are mandatory to be filled in the S104 spec.

    Parameters
    ----------
    output_file
        S104File object

    Returns
    -------
    data_file
        The S104File object created or updated by this function.


    """
    data_file = _get_S104File(output_file)
    root = data_file.root
    root.water_level_create()

    root.feature_information_create()
    group_f = root.feature_information
    group_f.feature_code_create()
    group_f.water_level_feature_dataset_create()

    water_level_feature_dataset = root.feature_information.water_level_feature_dataset

    water_level_height_info = water_level_feature_dataset.append_new_item()
    water_level_height_info.code = "waterLevelHeight"
    water_level_height_info.name = "Water level height"
    water_level_height_info.unit_of_measure = "meters"
    water_level_height_info.datatype = "H5T_FLOAT"
    water_level_height_info.fill_value = FILLVALUE_HEIGHT
    water_level_height_info.lower = "-99.99"
    water_level_height_info.upper = "99.99"
    water_level_height_info.closure = "closedInterval"

    water_level_trend_info = water_level_feature_dataset.append_new_item()
    water_level_trend_info.code = "waterLevelTrend"
    water_level_trend_info.name = "Water level trend"
    water_level_trend_info.unit_of_measure = ""
    water_level_trend_info.datatype = "H5T_ENUM"
    water_level_trend_info.fill_value = FILLVALUE_TREND
    water_level_trend_info.lower = "1"
    water_level_trend_info.upper = "3"
    water_level_trend_info.closure = "closedInterval"

    water_level_time_info = water_level_feature_dataset.append_new_item()
    water_level_time_info.code = "waterLevelTime"
    water_level_time_info.name = "Water level time"
    water_level_time_info.unit_of_measure = "DateTime"
    water_level_time_info.datatype = "H5T_C_S1"
    water_level_time_info.fill_value = ""
    water_level_time_info.lower = "19000101T000000Z"
    water_level_time_info.upper = "21500101T000000Z"
    water_level_time_info.closure = "closedInterval"

    utc_now = datetime.datetime.utcnow()
    root.issue_date = utc_now.strftime('%Y%m%d')
    root.issue_time = utc_now.strftime('%H%M%SZ')

    return data_file


def add_metadata(metadata: dict, data_file) -> S104File:
    """  Updates an S104File object based on input metadata.

    Parameters
    ----------
    data_file
        S104File object
    metadata
        a dictionary of metadata describing the data passed in,
        metadata should have the following key/value pairs:
            - "productSpecification": The product specification used to create
            this dataset.
            - "horizontalCRS": Horizontal Datum EPSG code.
            - "metadata": File name for the associated discovery metadata (xml)
            - "geographicIdentifier": Location of the data, ex: "Tampa Bay".
                An empty string ("") is the default.
            - "waterLevelHeightUncertainty": In (meters) arises from the
            hydrodynamic model, and the spatial interpolation method.
            The default, denoting a missing value, is -1.0.
            - "verticalUncertainty": Accuracy of vertical datum
                The default, denoting a missing value, is -1.0.
            - "horizontalPositionUncertainty": Accuracy of geolocation
            techniques, model grid accuracy. The default, denoting a missing
            value, is -1.0.
            - "timeUncertainty": Sensor accuracy, data time tagging accuracy
                The default, denoting a missing value, is -1.0.
            - "waterLevelTrendThreshold": Critical value used to determine
            steady water level trend. Units are meters/hour (m/hr).
            - "verticalCS": Vertical datum EPSG Code.
            - "verticalCoordinateBase":
                - 'Sea surface': 1
                - 'Vertical datum': 2
                - 'Sea Bottom': 3
            - "verticalDatumReference": For verticalCoordinateBase(2) only
                - 'S-100 vertical datum': 1
                - 'EPSG': 2
            - "verticalDatum":
                - 'meanLowWaterSprings': 1
                - 'meanLowerLowWaterSprings': 2
                - 'meanSeaLevel': 3
                - 'lowestLowWater': 4
                - 'meanLowWater': 5
                - 'lowestLowWaterSprings': 6
                - 'approximateMeanLowWaterSprings': 7
                - 'indianSpringLowWater': 8
                - 'lowWaterSprings': 9
                - 'approximateLowestAstronomicalTide': 10
                - 'nearlyLowestLowWater': 11
                - 'meanLowerLowWater': 12
                - 'lowWater': 13
                - 'approximateMeanLowWater': 14
                - 'approximateMeanLowerLowWater': 15
                - 'meanHighWater': 16
                - 'meanHighWaterSprings': 17
                - 'highWater': 18
                - 'approximateMeanSeaLevel': 19
                - 'highWaterSprings': 20
                - 'meanHigherHighWater': 21
                - 'equinoctialSpringLowWater': 22
                - 'lowestAstronomicalTide': 23
                - 'localDatum': 24
                - 'internationalGreatLakesDatum1985': 25
                - 'meanWaterLevel': 26
                - 'lowerLowWaterLargeTide': 27
                - 'higherHighWaterLargeTide': 28
                - 'nearlyHighestHighWater': 29
                - 'highestAstronomicalTide': 30
            - "verticalDatumReference":
                - 'S-100 Vertical datum': 1
                - 'EPSG code': 2
           - "commonPointRule":
                - 'average': 1
                - 'low': 2
                - 'high': 3
                - 'all': 4
           - "interpolationType": Interpolation method recommended for
           evaluation of the S100_GridCoverage.
                    - 'nearestneighbor': 1
                    - 'linear': 2
                    - 'quadratic': 3
                    - 'cubic': 4
                    - 'bilinear': 5
                    - 'biquadratic': 6
                    - 'bicubic': 7
                    - 'lostarea': 8
                    - 'barycentric': 9
                    - 'discrete': 10
            - "typeOfWaterLevelData:
                    - Observation: 1
                    - Astronomical prediction: 2
                    - Analysis or hybrid method: 3
                    - Hydrodynamic model hindcast: 4
                    - Hydrodynamic model forecast: 5
                    - Observed minus predicted: 6
                    - Observed minus analysis: 7
                    - Observed minus hindcast: 8
                    - Observed minus forecast: 9
                    - Forecast minus predicted: 10

            - "methodWaterLevelProduct": Brief description of tide gauge type,
            forecast method or model, etc.
            - "datetimeOfFirstRecord": Valid time of earliest value,
            'YYYYMMDDTHHMMSSZ'

    Returns
    -------
    data_file
        An S104File object updated by this function.

    """
    root = data_file.root

    water_level_feature = root.water_level
    water_level_feature.water_level_create()
    water_level_feature_instance_01 = water_level_feature.water_level.append_new_item()

    water_level_feature_instance_01.water_level_group_create()

    water_level_feature_instance_01.uncertainty_dataset_create()
    water_level_height_uncertainty = water_level_feature_instance_01.uncertainty_dataset.append_new_item()
    water_level_height_uncertainty.name = "waterLevelHeight"
    water_level_height_uncertainty.value = metadata["waterLevelHeightUncertainty"]

    water_level_feature.min_dataset_height = 0
    water_level_feature.max_dataset_height = 0
    water_level_feature_instance_01.time_record_interval = 0

    root.product_specification = S104File.PRODUCT_SPECIFICATION
    root.metadata = metadata["metadata"]
    root.horizontal_crs = metadata["horizontalCRS"]
    root.geographic_identifier = metadata["geographicIdentifier"]
    root.water_level_trend_threshold = metadata["waterLevelTrendThreshold"]
    root.vertical_coordinate_system = metadata["verticalCS"]
    root.vertical_coordinate_base = metadata["verticalCoordinateBase"]
    root.vertical_datum_reference = metadata["verticalDatumReference"]
    root.vertical_datum_epsg = metadata["verticalDatum"]
    water_level_feature.common_point_rule = metadata["commonPointRule"]
    water_level_feature.interpolation_type = metadata["interpolationType"]
    water_level_feature.time_uncertainty = metadata["timeUncertainty"]
    water_level_feature.vertical_uncertainty = metadata["verticalUncertainty"]
    water_level_feature.horizontal_position_uncertainty = metadata["horizontalPositionUncertainty"]
    water_level_feature.method_water_level_product = metadata["methodWaterLevelProduct"]
    water_level_feature_instance_01.type_of_water_level_data = metadata["typeOfWaterLevelData"]
    water_level_feature_instance_01.date_time_of_first_record = metadata["datetimeOfFirstRecord"]

    return data_file


def add_data_from_arrays(height: s1xx_sequence, trend, data_file, grid_properties: dict, datetime_value, data_coding_format) -> S104File:
    """  Updates an S104File object based on numpy array/h5py datasets.
        Calls :any:`create_s104` then fills in the HDF5 datasets with the
        supplied water level height and trend numpy.arrays.

        Raises an S104Exception if the shapes of the water level height and
        trend (if not None) grids are not equal.

        Parameters
        ----------
        height
            1d or 2d array containing water level heights
        trend
            1d or 2d array containing water level trends
        data_file
            S104File object
        datetime_value
            datetime object
        grid_properties
            a dictionary of metadata describing the grids passed in,
            metadata can have the following key/value pairs:
                - "maxx": West bound longitude
                - "minx": East bound longitude
                - "miny": South bound latitude
                - "maxy": North bound latitude
                - "cellsize_x": Only for DCF2, grid spacing longitude
                - "cellsize_y": Only for DCF2, grid spacing latitude
                - "nx": Only for DCF2, number of points longitudinal
                - "ny": Only for DCF2, number of points latitudinal
                - "latitude": Only for DCF3, latitude of nodes
                - "longitude": Only for DCF3, longitudes of nodes
                - "nodes": Only for DCF3, number of nodes
        data_coding_format
            - 'Time Series at fixed stations': 1
            - 'Regularly-Gridded arrays': 2
            - 'Ungeorectified Grid': 3
            - 'TIN': 7
            - 'Time Series at fixed stations (stationwise)': 8

        Returns
        -------
        data_file
            An S104File object updated by this function.

        """
    root = data_file.root
    water_level_feature = root.water_level
    water_level_feature_instance_01 = root.water_level.water_level[0]

    if data_coding_format == 2:
        water_level_feature.data_coding_format = data_coding_format
        water_level_feature_instance_01.start_sequence = "0,0"
        water_level_feature.sequencing_rule_scan_direction = "longitude, latitude"
        water_level_feature.sequencing_rule_type = 1
        water_level_feature_instance_01.grid_origin_longitude = grid_properties['maxx']
        water_level_feature_instance_01.grid_origin_latitude = grid_properties['miny']
        water_level_feature_instance_01.grid_spacing_longitudinal = grid_properties['cellsize_x']
        water_level_feature_instance_01.grid_spacing_latitudinal = grid_properties['cellsize_y']

        water_level_feature_instance_01.num_points_latitudinal = grid_properties['ny']
        water_level_feature_instance_01.num_points_longitudinal = grid_properties['nx']

    elif data_coding_format == 3:
        water_level_feature.data_coding_format = data_coding_format
        water_level_feature_instance_01.number_of_nodes = grid_properties['nodes']

        water_level_feature_instance_01.positioning_group_create()
        positioning = water_level_feature_instance_01.positioning_group
        positioning.geometry_values_create()
        geometry_values = positioning.geometry_values
        geometry_values.longitude = grid_properties['longitude']
        geometry_values.latitude = grid_properties['latitude']

    water_level_feature_instance_01.east_bound_longitude = grid_properties['minx']
    water_level_feature_instance_01.west_bound_longitude = grid_properties['maxx']
    water_level_feature_instance_01.south_bound_latitude = grid_properties['miny']
    water_level_feature_instance_01.north_bound_latitude = grid_properties['maxy']
    root.water_level.dimension = height.ndim

    water_level_feature.axis_names = numpy.array(["longitude", "latitude"])

    min_height = numpy.round(numpy.nanmin(height), decimals=2)
    max_height = numpy.round(numpy.nanmax(height), decimals=2)

    if min_height < water_level_feature.min_dataset_height:
        water_level_feature.min_dataset_height = min_height

    if max_height > water_level_feature.max_dataset_height:
        water_level_feature.max_dataset_height = max_height

    if numpy.ma.is_masked(height):
        height = height.filled(FILLVALUE_HEIGHT)

    height = numpy.round(height, decimals=2)
    trend.astype(int)

    if height.shape != trend.shape:
        raise S104Exception("Water level height & trend grids have different shapes")

    water_level_group_object = water_level_feature_instance_01.water_level_group.append_new_item()
    water_level_group_object.time_point = datetime_value

    water_level_group_object.values_create()
    grid = water_level_group_object.values
    grid.water_level_height = height
    grid.water_level_trend = trend

    return data_file


def update_metadata(data_file, grid_properties: dict, update_meta: dict) -> S104File:
    """  Updates an S104File object based on dynamic metadata.

          Parameters
          ----------
          data_file
              S104File object
          grid_properties
              a dictionary of metadata describing the dynamic data passed in,
              metadata can have the following key/value pairs:
                  - "maxx": West bound longitude
                  - "minx": East bound longitude
                  - "miny": South bound latitude
                  - "maxy": North bound latitude
                  - "cellsize_x": Only for DCF2, grid spacing longitude
                  - "cellsize_y": Only for DCF2, grid spacing latitude
                  - "nx": Only for DCF2, number of points longitudinal
                  - "ny": Only for DCF2, number of points latitudinal
                  - "latitude": Only for DCF3, latitude of nodes
                  - "longitude": Only for DCF3, longitudes of nodes
                  - "nodes": Only for DCF3, number of nodes
          update_meta
              a dictionary of dynamic metadata, metadata can have the following
              key/value pairs:
                  - "dateTimeOfLastRecord": Valid ISO 8601 time of latest value
                  - "numberOfGroups": Number of forecasts
                  - "numberOfTimes": Number of valid times
                  - "timeRecordInterval": Time between forecasts in seconds
                  - "num_instances": Number of water level feature instances

          Returns
          -------
          data_file
              An S104File object updated by this function.

          """
    root = data_file.root
    water_level_feature = root.water_level
    water_level_feature.num_instances = update_meta["num_instances"]
    water_level_feature_instance_01 = root.water_level.water_level[0]

    water_level_feature_instance_01.date_time_of_last_record = update_meta['dateTimeOfLastRecord']
    water_level_feature_instance_01.num_grp = update_meta['numberOfGroups']
    water_level_feature_instance_01.number_of_times = update_meta['numberOfTimes']
    water_level_feature_instance_01.time_record_interval = update_meta['timeRecordInterval']

    root.east_bound_longitude = grid_properties["minx"]
    root.west_bound_longitude = grid_properties["maxx"]
    root.south_bound_latitude = grid_properties["miny"]
    root.north_bound_latitude = grid_properties["maxy"]

    return data_file


def write_data_file(data_file):
    """  Writes file structure, metadata, data and closes S104File object."""

    data_file.write()
    data_file.flush()
    data_file.close()
