"""
"pytmc-debug" is a Qt interface that shows information about how pytmc
interprets TwinCAT3 .tmc files.
"""

import argparse
import logging
import pathlib
import sys

from qtpy import QtWidgets
from qtpy.QtCore import Qt, Signal

import pytmc

from .db import process

DESCRIPTION = __doc__
logger = logging.getLogger(__name__)


def _grep_record_names(text):
    if not text:
        return []

    records = [line.rstrip('{ ')
               for line in text.splitlines()
               if line.startswith('record')   # regular line
               or line.startswith('. record')  # linted line
               or line.startswith('X record')  # linted error line
               ]

    def split_rtyp(line):
        line = line.split('record(', 1)[1].rstrip('")')
        rtyp, record = line.split(',', 1)
        record = record.strip('"\' ')
        return f'{record} ({rtyp})'

    return [split_rtyp(record)
            for record in records
            ]


def _annotate_record_text(linter_results, record_text):
    if not record_text:
        return record_text
    if not linter_results or not (linter_results.warnings or
                                  linter_results.errors):
        return record_text

    lines = [([], line)
             for line in record_text.splitlines()]

    for item in linter_results.warnings + linter_results.errors:
        try:
            lint_md, line = lines[item['line'] - 1]
        except IndexError:
            continue

        if item in linter_results.warnings:
            lint_md.append('X Warning: {}'.format(item['message']))
        else:
            lint_md.append('X Error: {}'.format(item['message']))

    display_lines = []
    for lint_md, line in lines:
        if not lint_md:
            display_lines.append(f'. {line}')
        else:
            display_lines.append(f'X {line}')
            for lint_line in lint_md:
                display_lines.append(lint_line)

    return '\n'.join(display_lines)


class TmcSummary(QtWidgets.QMainWindow):
    '''
    pytmc debug interface

    Parameters
    ----------
    tmc : TmcFile
        The tmc file to inspect
    '''

    item_selected = Signal(object)

    def __init__(self, tmc, dbd, allow_no_pragma=False):
        super().__init__()
        self.tmc = tmc
        self.chains = {}
        self.incomplete_chains = {}
        self.records = {}

        records, self.exceptions = process(tmc, allow_errors=True,
                                           allow_no_pragma=allow_no_pragma)

        for record in records:
            if not record.valid:
                self.incomplete_chains[record.tcname] = record
                continue

            self.chains[record.tcname] = record
            try:
                record_text = record.render()
                linter_results = (pytmc.linter.lint_db(dbd, record_text)
                                  if dbd and record_text else None)
                record_text = _annotate_record_text(linter_results,
                                                    record_text)
            except Exception as ex:
                try:
                    record_text
                except NameError:
                    record_text = "Record not rendered"

                record_text = (
                    f'!! Linter failure: {ex.__class__.__name__} {ex}'
                    f'\n\n{record_text}'
                )

                logger.exception('Linter failure')

            self.records[record] = record_text

        self.setWindowTitle(f'pytmc-debug summary - {tmc.filename}')

        self._mode = 'chains'

        # Left part of the window
        self.left_frame = QtWidgets.QFrame()
        self.left_layout = QtWidgets.QVBoxLayout()
        self.left_frame.setLayout(self.left_layout)

        self.item_view_type = QtWidgets.QComboBox()
        self.item_view_type.addItem('Chains')
        self.item_view_type.addItem('Records')
        self.item_view_type.addItem('Chains w/o Records')
        self.item_view_type.currentTextChanged.connect(self._update_view_type)
        self.item_list = QtWidgets.QListWidget()
        self.item_list_filter = QtWidgets.QLineEdit()

        self.left_layout.addWidget(self.item_view_type)
        self.left_layout.addWidget(self.item_list_filter)
        self.left_layout.addWidget(self.item_list)

        # Right part of the window
        self.right_frame = QtWidgets.QFrame()
        self.right_layout = QtWidgets.QVBoxLayout()
        self.right_frame.setLayout(self.right_layout)

        self.record_text = QtWidgets.QTextEdit()
        self.record_text.setFontFamily('Courier New')
        self.chain_info = QtWidgets.QListWidget()
        self.config_info = QtWidgets.QTableWidget()

        self.right_layout.addWidget(self.record_text)
        self.right_layout.addWidget(self.chain_info)
        self.right_layout.addWidget(self.config_info)

        self.frame_splitter = QtWidgets.QSplitter()
        self.frame_splitter.setOrientation(Qt.Horizontal)
        self.frame_splitter.addWidget(self.left_frame)
        self.frame_splitter.addWidget(self.right_frame)

        self.top_splitter = self.frame_splitter
        if self.exceptions:
            self.error_list = QtWidgets.QTextEdit()
            self.error_list.setReadOnly(True)

            for ex in self.exceptions:
                self.error_list.append(f'({ex.__class__.__name__}) {ex}\n')

            self.error_splitter = QtWidgets.QSplitter()
            self.error_splitter.setOrientation(Qt.Vertical)
            self.error_splitter.addWidget(self.frame_splitter)
            self.error_splitter.addWidget(self.error_list)
            self.top_splitter = self.error_splitter

        self.setCentralWidget(self.top_splitter)
        self.item_list.currentItemChanged.connect(self._item_selected)

        self.item_selected.connect(self._update_config_info)
        self.item_selected.connect(self._update_chain_info)
        self.item_selected.connect(self._update_record_text)

        def set_filter(text):
            text = text.strip().lower()
            for idx in range(self.item_list.count()):
                item = self.item_list.item(idx)
                item.setHidden(bool(text and text not in item.text().lower()))

        self.item_list_filter.textEdited.connect(set_filter)

        self._update_item_list()

    def _item_selected(self, current, previous):
        'Slot - new list item selected'
        if current is None:
            return

        record = current.data(Qt.UserRole)
        if isinstance(record, pytmc.record.RecordPackage):
            self.item_selected.emit(record)
        elif isinstance(record, str):  # {chain: record}
            chain = record
            record = self.chains[chain]
            self.item_selected.emit(record)

    def _update_config_info(self, record):
        'Slot - update config information when a new record is selected'
        chain = record.chain

        self.config_info.clear()
        self.item_list_filter.setText('')
        self.config_info.setRowCount(len(chain.chain))

        def add_dict_to_table(row: int, d: dict):
            """
            Parameters
            ----------
            row : int
                Identify the row to configure.

            d : dict
                Dictionary of items to enter into the target row.
            """
            for key, value in d.items():
                key = str(key)
                if key not in columns:
                    columns[key] = max(columns.values()) + 1 if columns else 0
                # accumulate a list of entries to print in the chart. These
                # should only be printed after `setColumnCount` has been run
                if isinstance(value, dict):
                    yield from add_dict_to_table(row, value)
                else:
                    yield (
                        row, columns[key],
                        QtWidgets.QTableWidgetItem(str(value))
                    )

        columns = {}
        column_write_entries = []

        items = zip(chain.config['pv'], chain.item_to_config.items())
        for row, (pv, (item, config)) in enumerate(items):
            # info_dict is a collection of the non-field pragma lines
            info_dict = dict(pv=pv)
            if config is not None:
                info_dict.update(
                    {k: v for k, v in config.items() if k != 'field'})
                new_entries = add_dict_to_table(row, info_dict)
                column_write_entries.extend(new_entries)
                # fields is a dictionary exclusively contining fields
                fields = config.get('field', {})
                new_entries = add_dict_to_table(
                    row, {
                        f'field_{k}': v
                        for k, v in fields.items()
                        if k != 'field'
                    }
                )
                column_write_entries.extend(new_entries)

        # setColumnCount seems to need to proceed the setHorizontalHeaderLabels
        # in order to prevent QT from incorrectly drawing/labeling the cols
        self.config_info.setColumnCount(len(columns))
        self.config_info.setHorizontalHeaderLabels(list(columns))
        # finally print the column's entries
        for line in column_write_entries:
            self.config_info.setItem(*line)

        self.config_info.setVerticalHeaderLabels(
            list(item.name for item in chain.item_to_config))

    def _update_record_text(self, record):
        'Slot - update record text when a new record is selected'
        if self._mode.lower() == "chains w/o records":
            # TODO: a more helpful message oculd be useful
            self.record_text.setText("NOT GENERATED")
        else:
            self.record_text.setText(self.records[record])

    def _update_chain_info(self, record):
        'Slot - update chain information when a new record is selected'
        self.chain_info.clear()
        for chain in record.chain.chain:
            self.chain_info.addItem(str(chain))

    def _update_view_type(self, name):
        self._mode = name.lower()
        self._update_item_list()

    def _update_item_list(self):
        self.item_list.clear()
        if self._mode == 'chains':
            items = self.chains.items()
        elif self._mode == 'records':
            items = [
                (' / '.join(_grep_record_names(db_text)) or 'Unknown', record)
                for record, db_text in self.records.items()
            ]
        elif self._mode == "chains w/o records":
            items = self.incomplete_chains.items()
        else:
            return
        for name, record in sorted(items,
                                   key=lambda item: item[0]):
            item = QtWidgets.QListWidgetItem(name)
            item.setData(Qt.UserRole, record)
            self.item_list.addItem(item)


def create_debug_gui(tmc, dbd=None, allow_no_pragma=False):
    '''
    Show the results of tmc processing in a Qt gui.

    Parameters
    ----------
    tmc : TmcFile, str, pathlib.Path
        The tmc file to show.

    dbd : DbdFile, optional
        The dbd file to lint against.

    allow_no_pragma : bool, optional
        Look for chains that have missing pragmas.
    '''

    if isinstance(tmc, (str, pathlib.Path)):
        tmc = pytmc.parser.parse(tmc)

    if dbd is not None and not isinstance(dbd, pytmc.linter.DbdFile):
        dbd = pytmc.linter.DbdFile(dbd)

    return TmcSummary(tmc, dbd, allow_no_pragma=allow_no_pragma)


def build_arg_parser(parser=None):
    if parser is None:
        parser = argparse.ArgumentParser()

    parser.description = DESCRIPTION
    parser.formatter_class = argparse.RawTextHelpFormatter

    parser.add_argument(
        'tmc_file', metavar="INPUT", type=str,
        help='Path to .tmc file'
    )

    parser.add_argument(
        '-d', '--dbd',
        default=None,
        type=str,
        help=('Specify an expanded .dbd file for validating fields '
              '(requires pyPDB)')
    )

    parser.add_argument(
        '-a', '--allow-no-pragma',
        action='store_true',
        help='Show all items, even those missing pragmas (warning: slow)',
    )

    return parser


def main(tmc_file, *, dbd=None, allow_no_pragma=False):
    app = QtWidgets.QApplication([])
    interface = create_debug_gui(tmc_file, dbd,
                                 allow_no_pragma=allow_no_pragma)
    interface.show()
    sys.exit(app.exec_())
